﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.MappingService.Data;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public abstract class MapperBase
    {
        public Guid NetworkId { get; set; }

        public EntityReference Owner { get; set; }

        public PpmsReference References { get; set; }

        protected bool IsWithinContext { get; set; }
        public bool ForVaNetwork { get; set; }
        protected string NetworkShorthand { get; set; }

        protected readonly IPpmsContextHelper PpmsContextHelper;
        protected readonly IPpmsHelper PpmsHelper;

        private PpmsContext _context;
        protected PpmsContext Context
        {
            get
            {
                if (_context == null) _context = PpmsContextHelper.GetContextAsync().GetAwaiter().GetResult();
                return _context;
            }
            set
            {
                _context = value;
                // If context is set, use provided context for update activities
                IsWithinContext = _context != null;
            }
        }

        protected MapperBase(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
        {
            PpmsContextHelper = ppmsContextHelper;
            PpmsHelper = ppmsHelper;
        }
        
        public IEnumerable<SetStateRequest> EntityDelete<T>(IEnumerable<T> list)
        {
            var requests = new List<SetStateRequest>();
            var items = list as T[] ?? list.ToArray();
            if (!items.Any()) return requests;

            requests.AddRange(items.Select(EntityDelete).Where(request => request != null));

            return requests;
        }

        public SetStateRequest EntityDelete<T>(T entity)
        {
            return entity != null ? DeactivateEntity(entity) : null;
        }

        protected SetStateRequest DeactivateEntity<T>(T entity)
        {
            // Set default status code for all entities
            int statusCode = (int)Account_StatusCode.Inactive;
            // If entity is Account, set status code to Deactivate
            var account = entity as Entity;
            if (account != null && account.LogicalName == "account")
            {
                statusCode = (int)Account_StatusCode.Deactivated;
            }

            return PpmsHelper.SetEntityStatus(entity, (int)AccountState.Inactive, statusCode);
        }

        protected SetStateRequest ActivateEntity<T>(T entity)
        {
            return PpmsHelper.SetEntityStatus(entity, (int)AccountState.Active, (int)Account_StatusCode.Active);
        }

        protected bool IsChanged(string source, string target)
        {
            return !string.IsNullOrEmpty(source) && source != target;
        }

        protected void HandleException(string message)
        {
            throw new PpmsServiceException(message);
        }

        protected void HandleException(string message, string id)
        {
            HandleException($"{message} [Entity Id: {id}]");
        }

        protected IEnumerable<T> ConvertEntityList<T>(IEnumerable<object> entityList)
        {
            var objects = entityList as object[] ?? entityList.ToArray();
            if (!objects.Any()) return new List<T>();
            var list = objects.ToList();

            return list.ConvertAll(x => (T)x);
        }

        protected async Task SetNetworkByShorthand(string shorthand)
        {
            if (string.IsNullOrEmpty(shorthand)) return;

            var id = await PpmsHelper.GetNetworkIdByShorthand(shorthand);

            if (id.HasValue)
            {
                NetworkId = id.Value;
            }
        }

        protected async Task<Entity> GetProviderByCorrelationId(string correlationId, bool loadProperties = false)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                var id = new Guid(correlationId);
                var entity = context.AccountSet.FirstOrDefault(a => a.AccountId == id);
                if (loadProperties) LoadProviderProperties(entity, context);

                return entity;
            }
        }

        protected async Task<Entity> GetProviderByNpi(string npi, bool loadProperties = false)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                var entity = context.ppms_provideridentifierSet.FirstOrDefault(n => n.ppms_ProviderIdentifier == npi.Trim());
                if (entity == null) return null;

                context.LoadProperty(entity, "ppms_account_ppms_provideridentifier_Provider");

                if (entity.ppms_account_ppms_provideridentifier_Provider == null) return null;

                var provider = entity.ppms_account_ppms_provideridentifier_Provider;
                if (loadProperties) LoadProviderProperties(provider, context);

                return provider;
            }
        }

        protected EntityReference GetState(string state)
        {
            if (References != null && References.States != null)
            {
                var result = References.States.FirstOrDefault(s => s.Value == state.Trim());
                return new EntityReference("va_state", result.Id);
            }

            HandleException($"GetState: Unable to retrieve reference ({state}).");
            return null;

            //using (var context = await PpmsContextHelper.GetContextAsync())
            //{
            //    if (context != null)
            //    {
            //        return context.va_stateSet.FirstOrDefault(s => s.va_name == state);
            //    }

            //    HandleException("GetState: Unable to retrieve context.");
            //    return null;
            //}
        }

        protected EntityReference GetTaxonomy(string specialty)
        {
            Int16 state = 0;

            if (References != null && References.Specialties != null)
            {
                state++;
                var result = References.Specialties.FirstOrDefault(s => s.Value == specialty.Trim());
                if (result != null)
                    return new EntityReference("ppms_taxonomy", result.Id);
            }

            HandleException($"GetTaxonomy: Unable to retrieve taxonomy reference ({specialty},{state}).");
            return null;

            //using (var context = await PpmsContextHelper.GetContextAsync())
            //{
            //    if (context != null)
            //    {
            //        return context.ppms_taxonomySet.FirstOrDefault(s => s.ppms_SpecialtyCode == specialty);
            //    }

            //    HandleException("GetState: Unable to retrieve context.");
            //    return null;
            //}
        }

        protected void LoadProviderProperties(Entity entity, PpmsContext context)
        {
            if (entity == null || context == null) return;

            context.LoadProperty(entity, "ppms_account_ppms_provideridentifier_Provider");
            context.LoadProperty(entity, "ppms_account_ppms_providerservice");
            context.LoadProperty(entity, "ppms_account_ppms_boardcertification");
            context.LoadProperty(entity, "ppms_account_organizationauthorizedofficial");
            context.LoadProperty(entity, "ppms_account_ppms_otherprovideridentifier");
            context.LoadProperty(entity, "contact_customer_accounts");
            context.LoadProperty(entity, "ppms_account_providerlicensure");
            context.LoadProperty(entity, "ppms_account_ppms_othername");
            context.LoadProperty(entity, "ppms_account_ppms_providertaxonomy");
            context.LoadProperty(entity, "ppms_account_deascheduleprivilege");
            context.LoadProperty(entity, "ppms_account_providernetworkid");
        }


        public async Task ResetPpmsReferences()
        {
            References = new PpmsReference();

            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                // Load states
                var states = context.va_stateSet.Where(s => s.va_name != null);
                if (states != null)
                    References.States = states.ToList().ConvertAll(x => new PpmsEntityReference(x.Id, x.va_name));

                // Load speciaties
                var specs = context.ppms_taxonomySet.ToList();
                if (specs != null)
                    References.Specialties = specs.ConvertAll(x => new PpmsEntityReference(x.Id, x.ppms_SpecialtyCode));

                // Load VA provider relationships
                var relationships = context.ppms_vaproviderrelationshipSet.ToList();
                if (specs != null)
                    References.VaProviderRelationships = relationships.ConvertAll(x => new PpmsEntityReference(x.Id, x.ppms_name));
            }
        }
    }
}